function [] = deconv_multi(varargin)
%This is a high order script that allows you to run deconv_vstk on multiple
%directories, placed in the order of Root/Session/Channel/files
%Synatax:   [] = deconv_multi();
%Input:     'edgetaper' = taper the image edges according to the PSF.
%               Default = 0; off.
%           'deconv' = type of deconvolution used.  Default = lucy
%               Lucy-Richardson.  Other types: 'reg' = regularized,
%               'wiener' = wiener, and 'blind' = blind, based on maximum
%               likelihood algorithm.
%           'numit' = number of iteration for deconvolution, default = 10.
%               only applicable to blind and Lucy-Richardson.
%           'dampar' = the threshold deviation of the resulting image from
%               the original image (in terms of the standard deviation of
%               Poisson noise) below which damping occurs
%           'subsmpl' =  denotes subsampling and is used when the PSF is
%               given on a grid that is SUBSMPL times finer than the image.
%               Default is 1.
%           'initpsf' = the size of the initial psf generated for the blind
%               deconvolution.  The PSF restoration is affected strongly by
%               the size of the initial guess INITPSF and less by the
%               values it contains.
%           'cfx' = change file data types to preserve flux ratios. Default
%               = 0, off.  In this case, a 8 bit image will go to 16 and
%               back to 8bit, 16 bit image will go to 32bit then back to 16
%               for storage.
%           'save' = save the stacks as tiff files automatically. Default =
%               1.  0 = off.
%           'bksub' = background subtract.  Default = 0 (off).
%           'dir' = the root directory of the images
%           'psfdir' = the directory of PSFs

%Output:    no output at this moment

[initpsf,edgetapered,deconv_type,numit,dampar,subsmpl,sav,cfx,bk,dir_tmp,psf_dir] = parse(varargin);

%where are the files?
if isempty(dir_tmp)
    dir_tmp = uigetdir2('','Directory where your images are');    %get the directory
end
if isempty(psf_dir)
    psf_dir = uigetdir3('','Directory where the PSFs are located'); 
end

%now look at the root, if there are only files it will just pass it the
%root, if there are directories follow the above structure
dir_info = dir(dir_tmp);
dir_idx = [dir_info.isdir];   %grab all of the isdir numbers
dir_idx(1:2) = 0;               %the first two directories are not what we want
typeswitch = max(dir_idx);    %create the switch
curr_names = {dir_info.name};   %grab the all of the names in the root
dir_list = curr_names(dir_idx);    %the directory names in root

%prepare for saving
if sav
    mkdir(dir_tmp,'deconv');
    if bk
        mkdir(dir_tmp,'bksubtracted');
        mkdir(dir_tmp,'background');
    end
end
if typeswitch   %if directories
    for i = 1:size(dir_list,2)
        deconv_vstk('deconv',deconv_type,'edgetaper',edgetapered,'numit',numit,'initpsf',initpsf,'save',sav,'dampar',dampar,'subsmpl',subsmpl,...
            'cfx',cfx,'bksub',bk,'dir',[dir_tmp,filesep,dir_list{i}],'psfdir',psf_dir);
        %now save the deconv files in the same place
        if sav
            mkdir([dir_tmp,filesep,'deconv'],dir_list{i});
            mvdir([dir_tmp,filesep,dir_list{i},filesep,'deconv'],[dir_tmp,filesep,'deconv',filesep,dir_list{i}],0);
            if bk
                mkdir([dir_tmp,filesep,'bksubtracted'],dir_list{i});
                mvdir([dir_tmp,filesep,dir_list{i},filesep,'bksubtracted'],[dir_tmp,filesep,'bksubtracted',filesep,dir_list{i}],0);
                mkdir([dir_tmp,filesep,'background'],dir_list{i});
                mvdir([dir_tmp,filesep,dir_list{i},filesep,'background'],[dir_tmp,filesep,'background',filesep,dir_list{i}],0);
            end
        end       
    end
else    %no directories pass the root
    deconv_vstk('deconv',deconv_type,'edgetaper',edgetapered,'numit',numit,'initpsf',initpsf,'save',sav,'dampar',dampar,'subsmpl',subsmpl,...
        'cfx',cfx,'bksub',bk,'dir',dir_tmp,'psfdir',psf_dir);
end

%--------------------------------------------------------------------------
%subfunction to parse the inputs.
function [initpsf,edgetapered,deconv_type,numit,dampar,subsmpl,sav,cfx,bk,dir_tmp,psf_dir] = parse(input)

deconv_type = 'lucy';    %1=lucy-richardson(default), 2=regularized, 3=wiener, 4=blind.
edgetapered = 0;  %off.
numit = 10;     %default # of iteration = 10.
initpsf = 10;  %default = 10x10
sav = 1;       %default = save the file automatically
dampar = 0;     %default = 0 no dampling
subsmpl = 1;    %deault = 1 no subsampling
cfx = 0;     %by default off
bk = 0;      %background subtract off by default
dir_tmp = [];
psf_dir = [];

%Parse the input
if ~isempty(input)
    for i = 1:2:size(input,2)
        if ischar(input{1,i});
            switch input{1,i}
                case 'edgetaper'
                    edgetapered = input{1,i+1};
                case 'deconv'
                    if ischar(input{1,i+1})
                        deconv_type = input{1,i+1};
                    else
                        warning(['Your entered deconvolution type is not recognized, reverting to defalut type.']);
                    end
                case 'numit'
                    numit = input{1,i+1};
                case 'save'
                    sav = input{1,i+1};
                case 'dampar'
                    dampar = input{1,i+1};
                case 'subsmpl'
                    subsmpl = input{1,i+1};
                case 'initpsf'
                    initpsf = input{1,i+1};
                case 'cfx'
                    cfx = input{1,i+1};
                case 'bksub'
                    bk = input{1,i+1};
                case 'dir'
                    dir_tmp = input{1,i+1};
                case 'psfdir'
                    psf_dir = input{1,i+1};
                otherwise
                    warning(['Your input ',input{1,i},' is not recognized.']);
            end
        else
            error(['The parameters you entered is incorrect.  Please check help.']);
        end
    end
end